var base = require('./base');

class Modbus extends base
{
    constructor()
    {
        super();
        this.StartAddr = 0;
        this.NumberOfRegs = 1;
        this.datatype = "DT";
        this.Station = 1;
        this.writeData = [];
        this.writeBit = 0;
        this.selfProtocol = "MODBUS";
        this.transId = 1;
        this.protocIdent = 0;
        this.funcCode = 0;
        //Use odd max values for better calculations (word based)
        this.MAX_READ_REGS = 124;
        this.MAX_WRITE_REGS = 122;
        this.MAX_READ_BITS = 124;
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadRegisters - Generate the Modbus read command
     *
     * @param returns a buffer with the Modbus command
     ****************************************************************************/
    GenerateReadRegisters(startAddr = this.StartAddr, numbOfRegs = this.NumberOfRegs, station = this.Station, datatype = this.datatype)
    {
        //DT = read holding registers, FL or LD = Input registers
        this.funcCode = datatype === "DT" || datatype === "FL"  ? 3 : 4;
        //Read registers always have 12 bytes length
        let bufferLen = 12;
        let buf = Buffer.alloc(bufferLen)

        buf.writeUInt16BE(this.transId, 0);         //transaction Id (for multiple telegrams in parallel important otherwise can always be 1)
        buf.writeUInt16BE(this.protocIdent, 2);     //protocol identifier (always 0)
        buf.writeUInt16BE(bufferLen - 6, 4);        //Number of bytes within this telegram after this word
        buf.writeUInt8(station, 6);                 //Unit id
        buf.writeUInt8(this.funcCode, 7);           //Function code
        buf.writeUInt16BE(startAddr, 8);            //Start address
        buf.writeUInt16BE(numbOfRegs, 10);          //Number of regs to read

        return Buffer.from(buf, "hex");
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadSingleBit - Generate the Modbus read command
     *   
     * @param returns a buffer with the Modbus command
     ****************************************************************************/
    GenerateReadSingleBit(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype)
    {
        //R = read coil, L or Y or X = Input coil
        this.funcCode = datatype === "R" || datatype === "Y" ? 1 : 2;
        //Read always have 12 bytes length
        let bufferLen = 12;
        let buf = Buffer.alloc(bufferLen)

        buf.writeUInt16BE(this.transId, 0);         //transaction Id (for multiple telegrams in parallel important otherwise can always be 1)
        buf.writeUInt16BE(this.protocIdent, 2);     //protocol identifier (always 0)
        buf.writeUInt16BE(bufferLen - 6, 4);        //Number of bytes within this telegram after this word
        buf.writeUInt8(station, 6);                 //Unit id
        buf.writeUInt8(this.funcCode, 7);           //Function code
        buf.writeUInt16BE(startAddr, 8);       //Start address
        buf.writeUInt16BE(1, 10);                   //Number of bits to read

        return Buffer.from(buf, "hex");
    }

    /***************************************************************************/
    /**
     * @brief GenerateReadBits - Generate the Modbus read command
     *   
     * @param returns a buffer with the Modbus command
     ****************************************************************************/
    GenerateReadBits(startAddr = this.StartAddr, numbOfRegs = this.NumberOfRegs, station = this.Station, datatype = this.datatype)
    {
        //R = read coil, L or Y or X = Input coil
        this.funcCode = datatype === "R" || datatype === "Y" ? 1 : 2;
        //Read always have 12 bytes length
        let bufferLen = 12;
        let buf = Buffer.alloc(bufferLen)

        buf.writeUInt16BE(this.transId, 0);         //transaction Id (for multiple telegrams in parallel important otherwise can always be 1)
        buf.writeUInt16BE(this.protocIdent, 2);     //protocol identifier (always 0)
        buf.writeUInt16BE(bufferLen - 6, 4);        //Number of bytes within this telegram after this word
        buf.writeUInt8(station, 6);                 //Unit id
        buf.writeUInt8(this.funcCode, 7);           //Function code
        buf.writeUInt16BE(startAddr, 8);       //Start address
        buf.writeUInt16BE(numbOfRegs, 10);          //Number of bits to read

        return Buffer.from(buf, "hex");
    }

    /***************************************************************************/
    /**
     * @brief GenerateWriteRegisters - Generate the Modbus write command
     *   
     * @param returns a buffer with the Modbus command
     ****************************************************************************/
    GenerateWriteRegisters(startAddr = this.StartAddr, numbOfRegs = this.NumberOfRegs, station = this.Station, datatype = this.datatype, writedata = this.writeData)
    {
        //DT = read holding registers, FL or LD = Input registers
        this.funcCode = 16;
        //Read registers always have 12 bytes length
        let bufferLen = 13 + numbOfRegs * 2;
        let buf = Buffer.alloc(bufferLen);

        buf.writeUInt16BE(this.transId, 0);         //transaction Id (for multiple telegrams in parallel important otherwise can always be 1)
        buf.writeUInt16BE(this.protocIdent, 2);     //protocol identifier (always 0)
        buf.writeUInt16BE(bufferLen - 6, 4);        //Number of bytes within this telegram after this word
        buf.writeUInt8(station, 6);                 //Unit id
        buf.writeUInt8(this.funcCode, 7);           //Function code
        buf.writeUInt16BE(startAddr, 8);            //Start address
        buf.writeUInt16BE(numbOfRegs, 10);          //Number of registers to write
        buf.writeUInt8(numbOfRegs * 2, 12)            //Number of BYTES to write

        for (let i = 0; i < numbOfRegs; i++)
        {
            if (writedata[i] >= 0) {
                buf.writeUInt16BE(writedata[i], i * 2 + 13);
            }
            else {
                buf.writeInt16BE(writedata[i], i * 2 + 13);
            }
        }

        return Buffer.from(buf, "hex");
    }

    /***************************************************************************/
    /**
     * @brief GenerateWriteSingleBit - Generate the Modbus write command
     *   
     * @param returns a buffer with the Modbus command
     ****************************************************************************/
    GenerateWriteSingleBit(startAddr = this.StartAddr, station = this.Station, datatype = this.datatype, writebit = this.writeBit)
    {
        //DT = read holding registers, FL or LD = Input registers
        this.funcCode = 5;
        //Read registers always have 12 bytes length
        let bufferLen = 12;
        let buf = Buffer.alloc(bufferLen);

        let translatedAddr = this.h2d(startAddr);
        if (datatype === "R")
            translatedAddr += 2048; //Panasonic offset for R0

        buf.writeUInt16BE(this.transId, 0);         //transaction Id (for multiple telegrams in parallel important otherwise can always be 1)
        buf.writeUInt16BE(this.protocIdent, 2);     //protocol identifier (always 0)
        buf.writeUInt16BE(bufferLen - 6, 4);        //Number of bytes within this telegram after this word
        buf.writeUInt8(station, 6);                 //Unit id
        buf.writeUInt8(this.funcCode, 7);           //Function code
        buf.writeUInt16BE(translatedAddr, 8);       //Start address
        buf.writeUInt16BE(writebit, 10);            //Write value

        return Buffer.from(buf, "hex");
    }

    /***************************************************************************/
    /**
     * @brief DecodeReadRegisters - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeReadRegisters(response)
    {
        let result = {
            err: false,
            err_code: 0,
            int: [],
            uint: [],
            udint: [],
            dint: [],
            hex: [],
            real: [],
            string: ""
        }

        let i, j;
        let funcCodeOfResponse = response.readUInt8(7);

        //When the return function code = request function code + 0x80 then its an error
        if (funcCodeOfResponse === this.funcCode + 0x80)
        {
            result.err = true;
            result.err_code = this.d2h(response.readUInt8(8));
            return result;
        }

        let dataCount = response.readUInt8(8) / 2;
        let SingleData;

        for (i = 0; i < dataCount; i++)
        {
            //Extract one word. Save as ASCII hex string
            j = 9 + i * 2;
            SingleData = response.readUInt16BE(j);

            result.uint.push(SingleData);
            result.int.push(this.HexStringToINT(this.d2h(SingleData)));
            result.hex.push(this.d2h(SingleData));
            result.string += String.fromCharCode(response.readInt8(j + 1));
            result.string += String.fromCharCode(response.readInt8(j));
        }

        //combine DINT and UDINT
        let len = result.uint.length;
        for (i = 0; i < len; i = i + 2)
        {
            if (i + 1 !== len)
            {
                //Hex word must have 4 digits, if not add zeroes
                if (result.hex[i].length < 4)
                {
                    let zeroes = "";
                    for (let k = 0; k < 4-result.hex[i].length; k++)
                    {
                        zeroes += "0";
                    }

                    result.hex[i] = zeroes + result.hex[i];
                }
                let tmp = result.hex[i + 1] + result.hex[i];

                result.udint.push(this.h2d(tmp));
                result.dint.push(this.HexStringToDINT(tmp));
                result.real.push(this.CalculateIEEEReal(this.h2d(tmp)));
            }
        }

        return result;
    }

    /***************************************************************************/
    /**
     * @brief DecodeSingleBit - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeSingleBit(response)
    {
        let result = {
            err: false,
            err_code: 0,
            state: 0
        }
        
        let funcCodeOfResponse = response.readUInt8(7);

        //When the return function code = request function code + 0x80 then its an error
        if (funcCodeOfResponse === this.funcCode + 0x80)
        {
            result.err = true;
            result.err_code = this.d2h(response.readUInt8(8));
            return result;
        }

        let data = response.readInt8(9);
        result.state = data > 0 ? 1 : 0;

        return result;
    }

    /***************************************************************************/
    /**
     * @brief DecodeReadBits - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns an object with decoded values
     ****************************************************************************/
    DecodeReadBits(response, numOfBits = this.NumberOfRegs)
    {
        let result = {
            err: false,
            err_code: 0,
            state: []
        }

        let funcCodeOfResponse = response.readUInt8(7);

        //When the return function code = request function code + 0x80 then its an error
        if (funcCodeOfResponse === this.funcCode + 0x80)
        {
            result.err = true;
            result.err_code = this.d2h(response.readUInt8(8));
            return result;
        }

        let numberOfBytes = Math.floor(numOfBits / 8);

        if (numOfBits % 8 !== 0)
            numberOfBytes++;

        let data;
        let bitPosition;

        for (let i = 0; i < numberOfBytes; i++)
        {
            data = response.readInt8(i + 9);
            bitPosition = 0x01;
            for (let j = 0; j < 8; j++)
            {
                (data & bitPosition) > 0 ? result.state.push(1) : result.state.push(0);
                bitPosition = bitPosition << 1;
            }
        }

        return result;
    }

    /***************************************************************************/
    /**
     * @brief WriteResponseOK - Decode the answer from PLC
     *   
     * @param Response: the answer from PLC
     * @param returns true when the response was ok, otherwise false
     ****************************************************************************/
    WriteResponseOK(response)
    {
        let result = {
            err: false,
            err_code: 0,
            err_msg: ""
        };

        let funcCodeOfResponse = response.readUInt8(7);

        //When the return function code = request function code + 0x80 then its an error
        if (funcCodeOfResponse === this.funcCode + 0x80)
        {
            result.err = true;
            result.err_code = this.d2h(response.readUInt8(8));
            result.err_msg = "Unexpected respond";
        }

        return result;
    }
}

exports.Modbus = Modbus